import responseHandler from "../../utils/responseHandler.js";
import { PrismaClient as PrismaClient1 } from "../../../prisma/generated/client1/index.js";
import { PrismaClient as PrismaClient2 } from "../../../prisma/generated/client2/index.js";
import bcrypt from "bcrypt";
import signupMail from "../../common/emailTemplate/signUpMail.js";
import errorCodes from "../../utils/errorCodes.js";
import { signToken } from "../../utils/jwt.js";
import { createToken } from "../../utils/helper.js";
import { sendMail } from "../../common/sendMail/sendMail.js";
import { customResponse, failureResponse } from "../../utils/responseFn.js";

const prismaMasterDb = new PrismaClient1();

const prismaSubMasterDb = new PrismaClient2();

// AUTH //

// Create permissions //
const createPermissions = async (req, res) => {
  try {
    const { title } = req.body;

    // Check if a role with the same name already exists
    const existingPermission = await prismaMasterDb.permissions_list.findFirst({
      where: {
        title: title,
        deleted_at: null,
      },
    });

    // If the role already exists, return a conflict error
    if (existingPermission) {
      return responseHandler.error(
        res,
        errorCodes.ALREADY.code,
        errorCodes.message,
        400
      );
    }

    // Create a new role if the role name doesn't exist
    const newTitle = await prismaMasterDb.permissions_list.create({
      data: {
        title: title, // Store the new role name
        type: "permission",
        created_by: 1,
      },
    });

    if (!newTitle) {
      return responseHandler.error(res, {}, "Error while creating title.", 400);
    }
    // Send success response with the created role data
    return responseHandler.success(res, {}, "Title created successfully.", 200);
  } catch (err) {
    console.log(err, "error");
    return responseHandler.error(res, err);
  }
};

// fetch all permissions//
const getAllPermissions = async (req, res) => {
  try {
    let data;

    const {
      page_no = 1,
      number_of_rows = 10,
      status = 1,
      search_text = "",
    } = req.query;
    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(number_of_rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    // Calculate skip based on the current page
    const skip = (pageInt - 1) * limitInt;

    const allPermissionFetch = await prismaMasterDb.permissions_list.findMany({
      where: {
        deleted_at: null,
      },
      skip: skip,
      take: limitInt, // Limit the number of results per page
    });

    // Get total count for pagination info
    const totalCount = await prismaMasterDb.permissions_list.count({
      where: {
        deleted_at: null,
      },
    });
    // Calculate total pages
    const totalPages = Math.ceil(totalCount / limitInt);

    data = {
      permissions: allPermissionFetch,
      totalCount,
      totalPages,
      currentPage: pageInt,
      limitPerPage: limitInt,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

// delete permissions //

// edit permissions //

// Create roles //

const createRoles = async (req, res) => {
  try {
    const { name } = req.body;

    // Check if a role with the same name already exists
    const existingRole = await prismaMasterDb.roles.findUnique({
      where: {
        name: name,
        deleted_at: null,
        // Check if the role name already exists in the database
      },
    });

    // If the role already exists, return a conflict error
    if (existingRole) {
      return responseHandler.error(
        res,
        "error",
        "Roles is already exists.",
        400
      );
    }

    // Create a new role if the role name doesn't exist
    const newRole = await prismaMasterDb.roles.create({
      data: {
        name: name, // Store the new role name
      },
    });

    if (!newRole) {
      return responseHandler.error(res, {}, "Error while creating role.", 400);
    }
    // Send success response with the created role data
    return responseHandler.success(res, {}, "Role created successfully.", 200);
  } catch (err) {
    console.log(err, "eroeoroero");
    return responseHandler.error(res, err);
  }
};

// get all roles //

const allRoles = async (req, res) => {
  try {
    let data;

    const {
      page_no = 1,
      number_of_rows = 10,
      status = 1,
      search_text = "",
    } = req.query;
    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(number_of_rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    // Calculate skip based on the current page
    const skip = (pageInt - 1) * limitInt;

    const allRolesFetch = await prismaMasterDb.roles.findMany({
      where: {
        deleted_at: null,
        id: {
          not: 1,
        },
      },
      skip: skip,
      take: limitInt, // Limit the number of results per page
    });

    // Get total count for pagination info
    const totalCount = await prismaMasterDb.roles.count({
      where: {
        deleted_at: null,
        id: {
          not: 1,
        },
      },
    });
    // Calculate total pages
    const totalPages = Math.ceil(totalCount / limitInt);

    data = {
      roles: allRolesFetch,
      totalCount,
      totalPages,
      currentPage: pageInt,
      limitPerPage: limitInt,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.NOT_FOUND, err.message, 500);
  }
};

// delete roles //

// edit role //

// assign permission //
const assignPermissions = async (req, res) => {};

// create domain category//
const createDomainCategory = async (req, res) => {
  try {
    const { category_name } = req.body;

    // Check if a role with the same name already exists
    const existingCategory = await prismaSubMasterDb.ProjectCategory.findFirst({
      where: {
        category_name: category_name,
        deleted_at: null,
        // Check if the category name already exists in the database
      },
    });

    // If the role already exists, return a conflict error
    if (existingCategory) {
      return responseHandler.error(
        res,
        "error",
        "Category is already exists.",
        400
      );
    }

    // Create a new role if the role name doesn't exist
    const newCategory = await prismaSubMasterDb.ProjectCategory.create({
      data: {
        category_name: category_name,
        user_id: 1,
      },
    });

    if (!newCategory) {
      return responseHandler.error(res, {}, "Error while creating role.", 400);
    }
    // Send success response with the created role data
    return responseHandler.success(
      res,
      {},
      "Catgeory created successfully.",
      200
    );
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, err);
  }
};

const login = async (req, res) => {
  let data = {};
  try {
    const { email, password } = req.body;

    const userDetails = await prismaSubMasterDb.User.findUnique({
      where: { email },
    });

    if (!userDetails) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const compare = await bcrypt.compare(password, userDetails.password);

    if (!compare) {
      return responseHandler.error(
        res,
        errorCodes.PASSWORD,
        errorCodes.PASSWORD,
        400
      );
    }

    const userRole = await prismaMasterDb.roles.findUnique({
      where: {
        id: userDetails.role_id,
        deleted_at: null,
      },
    });

    let tokenData = {};
    tokenData[userRole.name] = {
      id: userDetails.id,
      email: userDetails.email,
    };

    const { id, first_name, last_name } = userDetails;

    data = { id, first_name, last_name, email };
    data.access_token = signToken(tokenData, "30d");
    // console.log(data, "checking data-->");
    return responseHandler.success(res, data, "Login success", 200);
  } catch (err) {
    console.log(err.message, "erereroer");
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

const fetchAllUsersAdmin = async (req, res) => {
  try {
    let data;
    const userData = req.user;

    const {
      page_no = 1,
      number_of_rows = 20,
      status = 1,
      search_text = "",
    } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findFirst({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(number_of_rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    // Calculate skip based on the current page
    const skip = (pageInt - 1) * limitInt;

    const allUserFetch = await prismaSubMasterDb.User.findMany({
      where: {
        deleted_at: null,
        UserInfo: {
          role_id: 2,
        },
      },
      select: {
        id: true,
        email: true,
        first_name: true,
        last_name: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
            organisation_id: true,
          },
        },
      },
      skip: skip,
      take: limitInt, // Limit the number of results per page
    });

    // Get total count for pagination info
    const totalCount = await prismaSubMasterDb.User.count({
      where: {
        deleted_at: null,
        UserInfo: {
          role_id: { not: 1 },
        },
      },
    });
    // Calculate total pages
    const totalPages = Math.ceil(totalCount / limitInt);

    data = {
      users: allUserFetch,
      totalCount,
      totalPages,
      currentPage: pageInt,
      limitPerPage: limitInt,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, err);
  }
};

const createUser = async (req, res) => {
  const userData = req.user;
  const { page_no, number_of_rows, role_id } = req.query;

  const ROLE = Object.getOwnPropertyNames(userData)[0];

  const existUser = await prismaSubMasterDb.User.findUnique({
    where: {
      id: userData[ROLE].id,
      deleted_at: null,
    },
  });

  if (!existUser) {
    return responseHandler.error(
      res,
      errorCodes.USER_NOT_EXIST,
      errorCodes.USER_NOT_EXIST,
      400
    );
  }

  let userDetails;

  const adminId = req.admin["ADMIN"]?.id;

  try {
    const {
      first_name,
      last_name,
      email,
      password,
      phone_no,
      country_code,
      country_iso_code,
      condition,
      verified,
    } = req.body;

    const existingUser = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
    });

    if (existingUser) {
      // const errorMessage =
      //   existingUser.is_verified === 0 && existingUser.status === 0
      //     ? errorLog.Pleaseverify
      //     : errorLog.DuplicateEmail;

      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        "User already exists",
        400
      );
    }

    const hashedPassword = await bcrypt.hash(password, 10);

    const existingRole = await prismaMasterDb.roles.findFirst({
      where: { name: "ORGANISATION", deleted_at: null },
    });

    if (!existingRole) {
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        "Role not found",
        400
      );
    }

    const newUser = await prismaSubMasterDb.User.create({
      data: {
        first_name,
        last_name,
        email,
        password: hashedPassword,
        phone_no,
        country_code,
        country_iso_code,
        condition,
        role_id: existingRole.id,
        is_verified: verified,
        status: "ACTIVE",
        create_by: adminId,
      },
    });

    if (!newUser) {
      return customResponse(res, false, 400, {}, errorLog.Usernotcreated);
    }

    // const token = createToken({ userDetails, role_name: "USER" }, "20m");

    // const url =
    //   process.env.NODE_ENV === "production"
    //     ? `http://${process.env.SERVER_FRONTEND_USER_URL}/auth/verify/${token}`
    //     : `http://${process.env.LOCAL_FRONTEND_URL}/verify/${token}`;

    // const msg = signupMail({ url, name: first_name });

    try {
      // await sendMail(email, msg.title, msg.description);
      // return customResponse(
      //   res,
      //   true,
      //   200,
      //   { email, token, userId: userDetails.id },
      //   "Email has been sent successfully. Please verify your email."
      // );

      return responseHandler.success(res, {}, "success", 200);
    } catch (error) {
      console.log(error, "eroeorore");
      return responseHandler.error(res, error, "err", 400);
    }
  } catch (error) {
    // console.log(error, "erroer checking");
    return responseHandler.error(res, error, "error", 500);
  }
};

const createOrganisation = async (req, res) => {
  try {
    const userData = req.user;
    const {
      first_name,
      last_name,
      email,
      password,
      phone_no,
      country_code,
      country_iso_code,
      subscription_id,
      condition,
    } = req.body;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findFirst({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
            user_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    const existingUser = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
      },
    });

    if (existingUser) {
      return customResponse(res, false, 400, {}, "Email already taken.");
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Find the "Organisation" role in the master DB
    // const existingRole = await prismaMasterDb.roles.findUnique({
    //   where: { id: 2, deleted_at: null },
    // });

    // if (!existingRole) {
    //   return
    // }

    // Create new user in the subscription database (prismaSubMasterDb)
    const createdUser = await prismaSubMasterDb.$transaction(async (prisma) => {
      const newUser = await prisma.User.create({
        data: {
          first_name,
          last_name,
          email,
          password: hashedPassword,
          phone_no,
          country_code,
          country_iso_code,
          condition,
          status: "ACTIVE",
          is_verified: true,
          create_by: existUser.id,
        },
      });

      await prisma.UserDetails.create({
        data: {
          user_id: newUser.id,
          organisation_id: newUser.id,
          role_id: 2,
        },
      });

      return newUser; // Return the new user details
    });

    if (!createdUser) {
      return customResponse(res, false, 400, {}, "Error while creating user.");
    }
    // Get the permissions for the new user (from master DB)
    const checkPermission = await prismaMasterDb.permissions_list.findUnique({
      where: {
        id: 1,
        deleted_at: null,
      },
    });

    if (!checkPermission) {
      return customResponse(res, false, 400, {}, "Permission not found.");
    }

    // Fetch subscription details, including features
    const subscriptionDetails = await prismaMasterDb.subscription.findUnique({
      where: {
        id: subscription_id,
        deleted_at: null,
      },
      include: {
        subscription_feature: {
          select: {
            id: true,
            status: true, // Include only the fields you need
            subscription_id: true,
            permission_id: true,
            limit: true,
          },
        },
      },
    });

    if (!subscriptionDetails) {
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        "Subscription details not found",
        400
      );
    }

    // Prepare insertedData for user_permissions and user_subscription
    let insertedData = subscriptionDetails.subscription_feature.map((item) => {
      return {
        user_id: createdUser.id,
        permission_id: item.permission_id,
        limit: item.limit,
      };
    });

    // Insert data into the user_permissions table in the master DB
    await prismaMasterDb.user_permissions.createMany({
      data: insertedData,
    });

    // Insert data into the user_subscription table in the master DB
    await prismaMasterDb.user_subscription.create({
      data: {
        user_id: createdUser.id,
        created_by_id: 1,
        subscription_id: subscription_id,
      },
    });

    // Return the success response
    return responseHandler.success(
      res,
      {},
      "Organisation created successfully.",
      200
    );
  } catch (error) {
    return failureResponse(res, error);
  }
};

const fetchAllSubscriptions = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findFirst({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    if (existUser.UserInfo?.role_id !== 1) {
      return customResponse(res, false, 401, {}, "Uanuthorized access.");
    }

    // Calculate skip based on the current page

    const allSubscriptions = await prismaMasterDb.subscription.findMany({
      where: {
        deleted_at: null,
      },
    });

    data = allSubscriptions;

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, err);
  }
};

export {
  createRoles,
  allRoles,
  createPermissions,
  createDomainCategory,
  login,
  getAllPermissions,
  // fetchAllUsers,
  fetchAllUsersAdmin,
  createUser,
  createOrganisation,
  fetchAllSubscriptions,
};
